// Service worker for Find It Elsewhere extension.
// Handles backend API communication as a fallback for popup direct calls.

const DEFAULT_BACKEND_URL = "https://shop.aantz.com";

async function getBackendUrl() {
  return new Promise((resolve) => {
    chrome.storage.sync.get({ backendUrl: DEFAULT_BACKEND_URL }, (items) => {
      resolve(items.backendUrl);
    });
  });
}

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.action === "lookupProduct") {
    handleLookup(msg.payload).then(sendResponse).catch((err) => {
      sendResponse({ error: err.message || "Backend request failed" });
    });
    return true; // async response
  }

  if (msg.action === "testConnection") {
    handleTestConnection().then(sendResponse).catch((err) => {
      sendResponse({ error: err.message || "Connection failed" });
    });
    return true;
  }
});

async function handleLookup(payload) {
  const backendUrl = await getBackendUrl();
  const resp = await fetch(`${backendUrl}/api/lookup`, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(payload),
  });
  if (!resp.ok) {
    const err = await resp.json().catch(() => ({}));
    throw new Error(err.error || `HTTP ${resp.status}`);
  }
  return resp.json();
}

async function handleTestConnection() {
  const backendUrl = await getBackendUrl();
  const resp = await fetch(`${backendUrl}/healthz`, { method: "GET" });
  if (!resp.ok) throw new Error(`HTTP ${resp.status}`);
  return resp.json();
}
