(function () {
  "use strict";

  const DEFAULT_BACKEND_URL = "https://shop.aantz.com";

  const VENDOR_COLORS = {
    "Walmart": "#0071CE",
    "Target": "#CC0000",
    "Home Depot": "#F96302",
    "Best Buy": "#0046BE",
    "Chewy": "#1C49C2",
    "Staples": "#CC0000",
    "Office Depot": "#D3222A",
    "B&H": "#00457C",
    "Newegg": "#D36B08",
  };

  let currentProductData = null;

  async function getBackendUrl() {
    return new Promise((resolve) => {
      chrome.storage.sync.get({ backendUrl: DEFAULT_BACKEND_URL }, (items) => {
        resolve(items.backendUrl);
      });
    });
  }

  function show(id) { document.getElementById(id).classList.remove("hidden"); }
  function hide(id) { document.getElementById(id).classList.add("hidden"); }

  function showError(msg) {
    hide("status");
    const el = document.getElementById("error");
    el.textContent = msg;
    show("error");
  }

  function setStatus(msg) {
    hide("error");
    show("status");
    document.querySelector("#status span").textContent = msg;
  }

  // Check if we're on an Amazon product page
  async function init() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (!tab || !tab.url) {
        showNotAmazon();
        return;
      }

      const isAmazon = /amazon\.(com|co\.uk|ca|de|fr|it|es)/.test(tab.url);
      const isProductPage = /\/(dp|gp\/product)\//.test(tab.url);

      if (!isAmazon || !isProductPage) {
        showNotAmazon();
        return;
      }

      // Extract product data from content script
      setStatus("Extracting product data...");
      chrome.tabs.sendMessage(tab.id, { action: "extractProductData" }, (response) => {
        if (chrome.runtime.lastError) {
          showError("Could not read page. Try refreshing the Amazon page.");
          return;
        }
        if (!response || !response.title) {
          showError("Could not extract product data from this page.");
          return;
        }
        currentProductData = response;
        showProductInfo(response);
        doLookup(response);
      });
    } catch (e) {
      showError("Error: " + e.message);
    }
  }

  function showNotAmazon() {
    hide("status");
    show("not-amazon");

    document.getElementById("manual-btn").addEventListener("click", () => {
      const input = document.getElementById("manual-input").value.trim();
      if (!input) return;
      currentProductData = { title: input, brand: "", model: "", source: "manual" };
      hide("not-amazon");
      showProductInfo(currentProductData);
      doLookup(currentProductData);
    });

    document.getElementById("manual-input").addEventListener("keydown", (e) => {
      if (e.key === "Enter") document.getElementById("manual-btn").click();
    });
  }

  function showProductInfo(data) {
    document.getElementById("product-title").textContent = data.title || "Unknown Product";
    const meta = [];
    if (data.brand) meta.push("Brand: " + data.brand);
    if (data.model) meta.push("Model: " + data.model);
    if (data.price) meta.push("Price: $" + data.price);
    if (data.asin) meta.push("ASIN: " + data.asin);
    document.getElementById("product-meta").textContent = meta.join(" \u00B7 ");
    show("product-info");
  }

  async function doLookup(productData) {
    setStatus("Searching other vendors...");

    try {
      const backendUrl = await getBackendUrl();
      const resp = await fetch(`${backendUrl}/api/lookup`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(productData),
      });

      if (!resp.ok) {
        const err = await resp.json().catch(() => ({}));
        throw new Error(err.error || `Backend returned HTTP ${resp.status}`);
      }

      const data = await resp.json();
      hide("status");
      renderResults(data);
    } catch (e) {
      if (e.message.includes("Failed to fetch") || e.message.includes("NetworkError")) {
        showError("Cannot reach backend. Is it running? Check extension options for the correct URL.");
      } else {
        showError("Search failed: " + e.message);
      }
    }
  }

  function updateSelectedCount() {
    const checked = document.querySelectorAll(".match-checkbox:checked");
    const btn = document.getElementById("open-selected-btn");
    btn.textContent = `Open Selected (${checked.length})`;
    btn.disabled = checked.length === 0;
  }

  function renderResults(data) {
    const container = document.getElementById("results");
    container.innerHTML = "";

    if (!data.matches || data.matches.length === 0) {
      container.innerHTML = '<div class="results-count">No matches found.</div>';
      show("results");
      return;
    }

    const countEl = document.createElement("div");
    countEl.className = "results-count";
    countEl.textContent = `${data.matches.length} match${data.matches.length !== 1 ? "es" : ""} found`;
    container.appendChild(countEl);

    for (const m of data.matches) {
      const color = VENDOR_COLORS[m.vendor] || "#b2bec3";
      const confClass = m.confidence >= 0.8 ? "confidence-high"
                      : m.confidence >= 0.5 ? "confidence-medium"
                      : "confidence-low";

      const card = document.createElement("div");
      card.className = "match-card";
      card.style.borderLeftColor = color;

      card.innerHTML = `
        <input type="checkbox" class="match-checkbox" data-url="${esc(m.url)}">
        <div class="match-vendor" style="color: ${color}">${esc(m.vendor)}</div>
        <div class="match-info">
          <div class="match-title" title="${esc(m.title)}">${esc(m.title)}</div>
          <div class="match-meta">${esc(m.availability || "Unknown")} &middot; ${esc(m.shipping || "Unknown")}</div>
        </div>
        <div class="match-price">${m.price != null ? "$" + m.price.toFixed(2) : "N/A"}</div>
        <span class="match-confidence ${confClass}">${Math.round(m.confidence * 100)}%</span>
        <a class="match-open" href="${esc(m.url)}" target="_blank" rel="noopener">Open</a>
      `;

      card.addEventListener("click", (e) => {
        if (e.target.tagName === "A" || e.target.tagName === "INPUT") return;
        const cb = card.querySelector(".match-checkbox");
        cb.checked = !cb.checked;
        card.classList.toggle("selected", cb.checked);
        updateSelectedCount();
      });

      const cb = card.querySelector(".match-checkbox");
      cb.addEventListener("change", () => {
        card.classList.toggle("selected", cb.checked);
        updateSelectedCount();
      });

      container.appendChild(card);
    }

    show("results");
    show("actions-bar");
  }

  function esc(str) {
    if (!str) return "";
    const div = document.createElement("div");
    div.textContent = str;
    return div.innerHTML;
  }

  // Open Selected - open all checked results in new tabs
  document.getElementById("open-selected-btn").addEventListener("click", () => {
    const checked = document.querySelectorAll(".match-checkbox:checked");
    checked.forEach((cb) => {
      const url = cb.getAttribute("data-url");
      if (url) window.open(url, "_blank");
    });
  });

  // Select All / Deselect All toggle
  document.getElementById("select-all-btn").addEventListener("click", () => {
    const boxes = document.querySelectorAll(".match-checkbox");
    const allChecked = Array.from(boxes).every((cb) => cb.checked);
    boxes.forEach((cb) => {
      cb.checked = !allChecked;
      cb.closest(".match-card").classList.toggle("selected", cb.checked);
    });
    document.getElementById("select-all-btn").textContent = allChecked ? "Select All" : "Deselect All";
    updateSelectedCount();
  });

  // Report mismatch
  document.getElementById("report-btn").addEventListener("click", () => {
    const btn = document.getElementById("report-btn");
    btn.textContent = "Thanks for the feedback!";
    btn.disabled = true;
    // In future: send anonymous feedback to backend
  });

  init();
})();
