// Content script for Amazon product pages.
// Extracts product data and responds to messages from popup/service worker.

(function () {
  "use strict";

  function extractASIN() {
    // Try URL patterns first
    const urlPatterns = [
      /\/dp\/([A-Z0-9]{10})/i,
      /\/gp\/product\/([A-Z0-9]{10})/i,
      /\/ASIN\/([A-Z0-9]{10})/i,
    ];
    for (const re of urlPatterns) {
      const m = window.location.href.match(re);
      if (m) return m[1];
    }
    // Try hidden input
    const input = document.querySelector('input[name="ASIN"]');
    if (input) return input.value;
    // Try data attribute
    const el = document.querySelector("[data-asin]");
    if (el && el.dataset.asin) return el.dataset.asin;
    return "";
  }

  function extractTitle() {
    const el = document.getElementById("productTitle");
    if (el) return el.textContent.trim();
    // Fallback selectors
    const alt = document.querySelector("#title, .product-title-word-break, h1.a-size-large");
    if (alt) return alt.textContent.trim();
    return "";
  }

  function extractPrice() {
    const selectors = [
      ".a-price .a-offscreen",
      "#priceblock_ourprice",
      "#priceblock_dealprice",
      "#priceblock_saleprice",
      ".apexPriceToPay .a-offscreen",
      "#corePrice_feature_div .a-offscreen",
      ".priceToPay .a-offscreen",
      "#price_inside_buybox",
      "#newBuyBoxPrice",
    ];
    for (const sel of selectors) {
      const el = document.querySelector(sel);
      if (el) {
        const text = el.textContent.trim();
        const m = text.match(/[\d,]+\.?\d{0,2}/);
        if (m) return parseFloat(m[0].replace(",", ""));
      }
    }
    return null;
  }

  function extractBrand() {
    // #bylineInfo link
    const byline = document.getElementById("bylineInfo");
    if (byline) {
      let text = byline.textContent.trim();
      // Remove "Visit the ... Store" or "Brand: ..."
      text = text.replace(/^Visit the\s+/i, "").replace(/\s+Store$/i, "");
      text = text.replace(/^Brand:\s*/i, "");
      if (text) return text;
    }
    // Product details table
    return extractFromDetailTable("brand") || "";
  }

  function extractModel() {
    return (
      extractFromDetailTable("model number") ||
      extractFromDetailTable("model name") ||
      extractFromDetailTable("item model number") ||
      ""
    );
  }

  function extractFromDetailTable(label) {
    // Technical details and product information tables
    const rows = document.querySelectorAll(
      "#productDetails_techSpec_section_1 tr, " +
      "#productDetails_detailBullets_sections1 tr, " +
      ".prodDetTable tr, " +
      "#detailBullets_feature_div li"
    );
    const labelLower = label.toLowerCase();
    for (const row of rows) {
      const th = row.querySelector("th, .a-text-bold, span.a-text-bold");
      const td = row.querySelector("td, span:not(.a-text-bold)");
      if (th && td) {
        if (th.textContent.toLowerCase().includes(labelLower)) {
          return td.textContent.trim();
        }
      }
    }
    // Also try detail bullets format
    const bullets = document.querySelectorAll("#detailBullets_feature_div .a-list-item");
    for (const li of bullets) {
      const spans = li.querySelectorAll("span span");
      if (spans.length >= 2) {
        if (spans[0].textContent.toLowerCase().includes(labelLower)) {
          return spans[1].textContent.trim();
        }
      }
    }
    return null;
  }

  function extractUPC() {
    return extractFromDetailTable("upc") || extractFromDetailTable("ean") || "";
  }

  function extractProductData() {
    return {
      source: "amazon",
      source_url: window.location.href,
      asin: extractASIN(),
      title: extractTitle(),
      brand: extractBrand(),
      model: extractModel(),
      price: extractPrice(),
      currency: "USD",
      upc: extractUPC(),
      images: [],
    };
  }

  // Listen for messages from popup or service worker
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.action === "extractProductData") {
      const data = extractProductData();
      sendResponse(data);
    }
    return true; // keep channel open for async
  });
})();
